/**
 * UTM Tracker - Script para capturar e preservar parâmetros UTM durante a jornada do usuário
 * Integração com UTMify para rastreamento de vendas
 * Formato compatível com Facebook Ads
 */

const UTMTracker = {
    // Parâmetros que serão capturados e preservados
    utmParams: [
        'utm_source', 
        'utm_medium', 
        'utm_campaign', 
        'utm_content', 
        'utm_term',
        'utm_id',
        'fbclid',
        'gclid',
        'src',
        'sck'
    ],
    
    // Duração padrão do cookie (30 dias)
    cookieDuration: 30,
    
    /**
     * Inicializa o rastreador de UTMs
     */
    init: function() {
        // Captura os parâmetros UTM da URL atual
        this.captureUTMParameters();
        
        // Adiciona os parâmetros UTM a todos os links internos
        this.attachUTMsToLinks();
        
        // Prepara o formulário para incluir parâmetros UTM
        this.setupFormTracking();
        
        console.log('UTM Tracker inicializado');
    },
    
    /**
     * Captura os parâmetros UTM da URL e os armazena em cookies
     */
    captureUTMParameters: function() {
        const urlParams = new URLSearchParams(window.location.search);
        let utmCaptured = false;
        
        // Verifica cada parâmetro UTM na URL e o armazena em um cookie
        this.utmParams.forEach(param => {
            if (urlParams.has(param)) {
                const value = urlParams.get(param);
                this.setCookie(param, value);
                utmCaptured = true;
                console.log(`Parâmetro ${param} capturado: ${value}`);
            }
        });
        
        // Se nenhum parâmetro UTM foi encontrado, mas há um referrer, armazena-o como utm_source
        if (!utmCaptured && !this.getCookie('utm_source') && document.referrer) {
            try {
                const referrerUrl = new URL(document.referrer);
                // Não armazena o referrer se for do mesmo domínio
                if (referrerUrl.hostname !== window.location.hostname) {
                    const refSource = referrerUrl.hostname.replace('www.', '');
                    this.setCookie('utm_source', refSource);
                    
                    // Para o padrão FB, define FB como source se vier do facebook ou instagram
                    if (refSource.includes('facebook') || refSource.includes('instagram')) {
                        this.setCookie('utm_source', 'FB');
                    }
                    
                    console.log(`Referrer capturado como utm_source: ${refSource}`);
                }
            } catch (e) {
                console.log('Erro ao processar referrer:', e);
            }
        }
        
        // Detecta FBCLID e define utm_source como FB se não estiver definido
        if (urlParams.has('fbclid') && !this.getCookie('utm_source')) {
            this.setCookie('utm_source', 'FB');
            console.log('FBCLID detectado, utm_source definido como FB');
        }
    },
    
    /**
     * Adiciona os parâmetros UTM armazenados a todos os links internos
     */
    attachUTMsToLinks: function() {
        // Seleciona todos os links na página
        const links = document.querySelectorAll('a');
        
        links.forEach(link => {
            // Ignora links sem href ou links externos/mailto/tel
            if (!link.href || 
                link.href.startsWith('mailto:') || 
                link.href.startsWith('tel:') || 
                link.href.startsWith('#') || 
                link.href.startsWith('javascript:')) {
                return;
            }
            
            try {
                const url = new URL(link.href);
                
                // Apenas processa links do mesmo domínio
                if (url.hostname === window.location.hostname) {
                    // Obtém os parâmetros UTM armazenados
                    const utmParams = this.getAllUTMParams();
                    
                    // Adiciona os parâmetros à URL
                    Object.keys(utmParams).forEach(key => {
                        if (utmParams[key]) {
                            url.searchParams.set(key, utmParams[key]);
                        }
                    });
                    
                    // Atualiza o link
                    link.href = url.toString();
                }
            } catch (e) {
                console.log('Erro ao processar link:', e);
            }
        });
    },
    
    /**
     * Configura o rastreamento para formulários
     */
    setupFormTracking: function() {
        const forms = document.querySelectorAll('form');
        
        forms.forEach(form => {
            form.addEventListener('submit', (event) => {
                const utmParams = this.getAllUTMParams();
                
                // Adiciona campos ocultos com os valores UTM
                Object.keys(utmParams).forEach(key => {
                    if (utmParams[key]) {
                        // Verifica se o campo já existe
                        let input = form.querySelector(`input[name="${key}"]`);
                        
                        if (!input) {
                            // Cria um novo campo oculto
                            input = document.createElement('input');
                            input.type = 'hidden';
                            input.name = key;
                            form.appendChild(input);
                        }
                        
                        // Define o valor do campo
                        input.value = utmParams[key];
                    }
                });
            });
        });
    },
    
    /**
     * Define um cookie com o nome e valor especificados
     */
    setCookie: function(name, value) {
        const date = new Date();
        date.setTime(date.getTime() + (this.cookieDuration * 24 * 60 * 60 * 1000));
        const expires = "expires=" + date.toUTCString();
        document.cookie = name + "=" + encodeURIComponent(value) + ";" + expires + ";path=/";
    },
    
    /**
     * Obtém o valor de um cookie pelo nome
     */
    getCookie: function(name) {
        const nameEQ = name + "=";
        const ca = document.cookie.split(';');
        for (let i = 0; i < ca.length; i++) {
            let c = ca[i];
            while (c.charAt(0) === ' ') {
                c = c.substring(1);
            }
            if (c.indexOf(nameEQ) === 0) {
                return decodeURIComponent(c.substring(nameEQ.length, c.length));
            }
        }
        return null;
    },
    
    /**
     * Retorna todos os parâmetros UTM como um objeto
     */
    getAllUTMParams: function() {
        const params = {};
        
        this.utmParams.forEach(param => {
            const value = this.getCookie(param);
            if (value) {
                params[param] = value;
            }
        });
        
        return params;
    },
    
    /**
     * Adiciona parâmetros UTM a uma URL
     */
    addUTMParamsToUrl: function(url) {
        try {
            const urlObj = new URL(url);
            const utmParams = this.getAllUTMParams();
            
            Object.keys(utmParams).forEach(key => {
                if (utmParams[key]) {
                    urlObj.searchParams.set(key, utmParams[key]);
                }
            });
            
            return urlObj.toString();
        } catch (e) {
            console.log('Erro ao adicionar UTMs à URL:', e);
            return url;
        }
    },
    
    /**
     * Retorna uma string de consulta com todos os parâmetros UTM
     */
    getUTMQueryString: function() {
        const utmParams = this.getAllUTMParams();
        const params = new URLSearchParams();
        
        Object.keys(utmParams).forEach(key => {
            if (utmParams[key]) {
                params.append(key, utmParams[key]);
            }
        });
        
        return params.toString();
    }
};

// Inicializa o rastreador quando o documento estiver pronto
document.addEventListener('DOMContentLoaded', function() {
    UTMTracker.init();
});
